// $Id: HTTPRandomAccessFile3.java,v 1.2 2004/10/02 20:56:04 caron Exp $
/*
 * Copyright 1997-2000 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * HTTPRandomAccessFile.java.
 * @author John Caron, based on work by Donald Denbo
 */

package ucar.unidata.io.http;

import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.methods.*;

import java.io.FileNotFoundException;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.InputStream;

/**
 * HTTPRandomAccessFile.java, using jakarta commons HttpClient.
 * @author John Caron
 */

public class HTTPRandomAccessFile3 extends ucar.unidata.io.RandomAccessFile {
  static public int defaultHTTPBufferSize = 20000;
  static public int timeoutMsecs = 5000;

  private String url;
  private HttpClient client;
  private long total_length = 0;

  private boolean debug = false, debugDetails = false;

  public HTTPRandomAccessFile3(String url) throws IOException {
    this( url, defaultHTTPBufferSize);
    location = url.toString();
  }

  public HTTPRandomAccessFile3(String url, int bufferSize) throws IOException {
    super( bufferSize);
    file = null;
    this.url = url;
    location = url.toString();

    client = new HttpClient();
    client.setConnectionTimeout(timeoutMsecs); //establish a connection within this time

    HttpMethod method = new HeadMethod(url);
    method.setFollowRedirects(true);
    method.setStrictMode(false);

    doConnect( method);

    String s = method.getResponseHeader("Content-Length").getValue();
    total_length = Integer.parseInt( s);

    method.releaseConnection();
    // header[0] = new NVPair("User-Agent", "HTTPnetCDF;"); ??
  }

  private void doConnect(HttpMethod method) throws IOException {

    // Execute the method.
    int statusCode = -1;

    // We will retry up to 3 times.
    for (int attempt = 0; statusCode == -1 && attempt < 3; attempt++) {
      try {
        statusCode = client.executeMethod(method);

      } catch (HttpRecoverableException e) {
         if (attempt == 2) throw new IOException( e.getMessage());
         System.err.println("A recoverable exception occurred, retrying." + e.getMessage());
      }

    }

    if (statusCode == 404)
      throw new FileNotFoundException( url+" "+method.getStatusLine());
    if (statusCode >= 300)
      throw new IOException( url+" "+method.getStatusLine());

    if (debugDetails) {

        //write out the request headers
        System.out.println("*** Request ***");
        System.out.println("Request Path: " + method.getPath());
        System.out.println("Request Query: " + method.getQueryString());
        Header[] requestHeaders = method.getRequestHeaders();
        for (int i=0; i<requestHeaders.length; i++){
            System.out.print(requestHeaders[i]);
        }

        //write out the response headers
        System.out.println("*** Response ***");
        System.out.println("Status Line: " + method.getStatusLine());
        Header[] responseHeaders = method.getResponseHeaders();
        for (int i=0; i<responseHeaders.length; i++){
            System.out.print(responseHeaders[i]);
        }
    }

  }


  protected int read_(long pos, byte[] buff, int off, int len) throws IOException {
    long end = pos + len - 1;
    if (end >= total_length)
      end = total_length - 1;

    if (debug) System.out.println(" HTTPRandomAccessFile bytes="+pos+"-"+end+": ");

    HttpMethod method = new GetMethod(url);  // LOOK: can we resuse?
    method.setFollowRedirects(true);
    method.setStrictMode(false);

    method.setRequestHeader("Range", "bytes="+pos+"-"+end);
    doConnect( method);

    String s = method.getResponseHeader("Content-Length").getValue();
    int readLen = Integer.parseInt( s);

    InputStream is = method.getResponseBodyAsStream();
    readLen = copy( is, buff, readLen);
    //if (debug) System.out.println(" HTTPRandomAccessFile2 readLen="+readLen);

    method.releaseConnection();

    return readLen;
  }

  private int copy(InputStream in, byte[] buff, int want) throws IOException {
    int done = 0;
    while (true) {
      int bytesRead = in.read( buff, done, want);
      if (bytesRead == -1) break;
      done += bytesRead;
      want -= bytesRead;
    }
    return done;
  }

  public long length( ) throws IOException {
    long fileLength = total_length;
    if( fileLength < dataEnd )
      return dataEnd;
    else
      return fileLength;
  }


  /**
   * override the rest of the RandomAccessFile public methods
   */
  public void close() {
    client = null;
  }

  public FileDescriptor getFD() {
    return null;
  }

}

