// $Id: DateUnit.java,v 1.2 2004/10/20 23:23:15 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ucar.nc2.units;

import ucar.units.*;
import java.util.Date;

/**
 * Handles dates, represented as "n units of time since reference date" eg
 *  "1203 days since 1970-01-01 00:00:00".
 *
 *  Use SimpleUnit.factory() to create one.
 *
 * @author caron
 * @version $Revision: 1.2 $ $Date: 2004/10/20 23:23:15 $
 */

public class DateUnit extends SimpleUnit {
  static private java.text.SimpleDateFormat dateFormat, dateOnlyFormat;
  static {
    dateFormat = new java.text.SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    dateFormat.setTimeZone(java.util.TimeZone.getTimeZone("GMT")); // same as UTC

    dateOnlyFormat = new java.text.SimpleDateFormat("yyyy-MM-dd");
    dateOnlyFormat.setTimeZone(java.util.TimeZone.getTimeZone("GMT")); // same as UTC
  }

  /** Create a Date from this String.
   * @param unitString a udunit string.
   *   <pre>(number) (units) since [-]Y[Y[Y[Y]]]-MM-DD[(T| )hh[:mm[:ss[.sss*]]][ [+|-]hh[[:]mm]]]</pre>
   * @return Date or null if not date unit.
   */
  static public Date getStandardDate(String unitString) {
    // UnitFormat format = UnitFormatManager.instance();
    Unit uu = null;
    try {
      uu = format.parse(unitString);
    } catch (Exception e) {
      return null; // not valid unit
    }

    if (!(uu instanceof TimeScaleUnit)) return null; // not date unit

    TimeScaleUnit tsu = (TimeScaleUnit) uu;
    Date origin = tsu.getOrigin();

    if (!(tsu.getUnit() instanceof ScaledUnit)) return null; // not date unit

    ScaledUnit offset = (ScaledUnit) tsu.getUnit();
    double value = offset.getScale();
    /* System.out.println("offset= "+offset);

    try {
      value = tsu.getUnit().convertTo(offset.getScale(), SI.SECOND);
      System.out.println("value= "+value);
    } catch (Exception e) {
      return null;
    } */

    Date result = new Date( origin.getTime() + (long)(1000*value));
    // System.out.println("result= "+result);

    return result;
  }


  /** Return standard GMT date format */
   public static String getStandardDateString(Date date) {
     return dateFormat.format(date);
   }

  /** Return standard GMT date format; show date only, not time  */
   public static String getStandardDateOnlyString(Date date) {
     return dateOnlyFormat.format(date);
   }


  ////////////////////////////////////////////////////////////////////////

  DateUnit(ucar.units.Unit uu) {
    super( uu);
  }

  /** Create a Date from this unit and its current value.
   * @return Date or null if not a time unit.
   */
  public Date getDateOrigin() {
    if (!(uu instanceof TimeScaleUnit)) return null;
    TimeScaleUnit tu = (TimeScaleUnit) uu;
    return tu.getOrigin();
  }

  /** Create a Date from this unit and the given value.
   * @param value of time
   * @return Date or null if not time unit.
   */
  public Date getStandardDate(double value) {

    double secs;
    try {
      secs = uu.convertTo( value, dateUnit);
      if (debugParse) System.out.println(value+" "+ getName()+ "= " +secs+" secs since the epoch");
    } catch (Exception e) {
      if (debugParse) System.out.println("getStandardDateString got Exception " +e);
      return null;
    }
    java.util.Date date = new java.util.Date();
    date.setTime((long)(1000.0* secs));

    return date;
  }


  /** Make a standard GMT string representation from this unit and given value.
   * @param value of time
   * @return String or null if not time unit.
   */
  public String getStandardDateString(double value) {
    java.util.Date date = getStandardDate( value);
    if (date == null) return null;
    return dateFormat.format(date);
  }

  public String toString() {
    return getStandardDateString(1.0);
  }

  /** testing */
  public static void main(String[] args) throws Exception {
    long msec = 0;
    Date d = new Date(msec);
    System.out.println(msec +" = "+getStandardDateString(d));

    msec = -60L * 60 * 24 * 1000 * 365 * 1972;
    d = new Date(msec);
    System.out.println(msec +" = "+getStandardDateString(d));

    msec = Long.MAX_VALUE;
    d = new Date(msec);
    System.out.println(msec +" = "+getStandardDateString(d));

    msec = Long.MIN_VALUE;
    d = new Date(msec);
    System.out.println(msec +" = "+getStandardDateString(d));
  }

}


/**
 * $Log: DateUnit.java,v $
 * Revision 1.2  2004/10/20 23:23:15  caron
 * add nexrad2 iosp
 *
 * Revision 1.1  2004/09/11 00:04:21  caron
 * *** empty log message ***
 *
 * Revision 1.1  2004/07/12 23:40:19  caron
 * 2.2 alpha 1.0 checkin
 *
 *
 */