// $Id: VerticalCT.java,v 1.3 2004/12/07 01:29:30 caron Exp $
/*
 * Copyright 2002-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package ucar.nc2.dataset;

import ucar.unidata.geoloc.vertical.*;


/**
 * A Vertical CoordinateTransform is a function CT: (GeoZ) -> Height or Pressure.
 * Typically it may be dependent also on X,Y and/or Time. CT: (X,Y,GeoZ,Time) -> Height or Pressure.
 * This class just records the transformation parameters. The mathematical transformation itself is
 * delegated to a class implementing ucar.unidata.geoloc.vertical.VerticalTransform;
 *
 * @author caron
 * @version $Revision: 1.3 $ $Date: 2004/12/07 01:29:30 $
 */

public class VerticalCT extends CoordinateTransform {
   private VerticalTransform vt;
   private VerticalCT.Type type;

  /**
   * Create a Vertical Coordinate Transform.
   * @param name name of transform, must be unique within the dataset.
   * @param authority naming authority.
   * @param type type of vertical transform
   */
  public VerticalCT (String name, String authority, VerticalCT.Type type) {
    super( name, authority, TransformType.Vertical);
    this.type = type;
  }

  /** get the Vertical Transform type */
  public VerticalCT.Type getVerticalTransformType() { return type; }
  /** get the Vertical Transform function */
  public VerticalTransform getVerticalTransform() { return vt; }
  /** get the Vertical Transform function */
  public void setVerticalTransform(VerticalTransform vt ) { this.vt = vt; }

  //////////////////////////////////////////////////////////////////////////////////////////////////
  // inner class VerticalCT.Type
  private static java.util.HashMap hash = new java.util.HashMap(10);

  public static class Type {
    // These are from CF-1.0
    public final static Type Sigma = new Type("atmosphere_sigma");
    public final static Type HybridSigmaPressure = new Type("atmosphere_hybrid_sigma_pressure");
    public final static Type HybridHeight = new Type("atmosphere_hybrid_height");
    public final static Type Sleve = new Type("atmosphere_sleve");
    public final static Type OceanSigma = new Type("ocean_sigma");
    public final static Type OceanS = new Type("ocean_s");
    public final static Type OceanSigmaZ = new Type("ocean_sigma_z");
    public final static Type OceanDoubleSigmaZ = new Type("ocean_double_sigma");

    // WRF
    public final static Type WRFEta = new Type("WRFEta");

    private String name;
    private Type(String s) {
      this.name = s;
      hash.put( s, this);
    }

    /**
     * Find the VerticalCT.Type that matches this name.
     * @param name
     * @return VerticalCT.Type or null if no match.
     */
    public static Type getType(String name) {
      if (name == null) return null;
      return (Type) hash.get( name);
    }
    public String toString() { return name; }
  }

}

/* Change History:
   $Log: VerticalCT.java,v $
   Revision 1.3  2004/12/07 01:29:30  caron
   redo convention parsing, use _Coordinate encoding.

   Revision 1.2  2004/09/22 18:43:02  caron
   move common to ucar.unidata; projections use Parameter, no tAttribute


   Revision 1.5  2004/07/29 22:22:54  lind
   Add types for WRF transforms.

   Revision 1.4  2003/09/19 00:11:41  caron
   debugging and javadoc fixes

   Revision 1.3  2003/07/12 23:30:07  caron
   fix javadoc

   Revision 1.2  2003/07/12 23:08:53  caron
   add cvs headers, trailers

*/