// $Id: EnhancementsImpl.java,v 1.6 2004/12/07 02:43:21 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ucar.nc2.dataset;

import ucar.nc2.*;

import java.util.*;

/**
/**
 * Implementation of Enhancements for coordinate systems and standard attribute handling.
 * @author caron
 * @version $Revision: 1.6 $ $Date: 2004/12/07 02:43:21 $
 */
class EnhancementsImpl implements Enhancements {
  private Variable orgVar;
  private String desc, units;
  private ArrayList coordSys = new ArrayList();

  /**
   * Constructor when there's no underlying, existing Variable.
   * You can access units, description and coordSys.
   * All missing and scale/offset flags are false.
   */
  public EnhancementsImpl( String units, String desc) {
    this.units = units;
    this.desc = desc;
  }

  /**
   * Constructor.
   * @param orgVar the original Variable to decorate.
   */
  public EnhancementsImpl( Variable orgVar) {
    this.orgVar = orgVar;
  }

  /**
   * Get the list of Coordinate Systems for this variable.
   * Normally this is empty unless you use ucar.nc2.dataset.NetcdfDataset.
   * @return list of type ucar.nc2.dataset.CoordinateSystem; may be empty not null.
   */
  public List getCoordinateSystems() {return coordSys; }

  /** Add a CoordinateSystem to the dataset. */
  public void addCoordinateSystem( CoordinateSystem cs){
    coordSys.add(cs);
  }

  /** Get the original variable */
  public Variable getOriginalVariable() { return orgVar; }

  /** Set the Description for this Variable. */
  public void setDescription(String desc) { this.desc = desc; }

  /** Get the description of the Variable.
   *  Default is to use "long_name" attribute value. If not exist, use "description" attribute value.
   *  If not exist, use name.
   */
  public String getDescription() {
    if ((desc == null) && (orgVar != null)) {
      Attribute att = orgVar.findAttributeIgnoreCase( "long_name");
      if ((att != null) && att.isString())
        desc = att.getStringValue();

      if (desc == null) {
        att = orgVar.findAttributeIgnoreCase( "description");
        if ((att != null) && att.isString())
          desc = att.getStringValue();
      }

      if (desc == null) {
        att = orgVar.findAttributeIgnoreCase( "title");
        if ((att != null) && att.isString())
          desc = att.getStringValue();
      }

      if (desc == null) {
        att = orgVar.findAttributeIgnoreCase( "standard_name");
        if ((att != null) && att.isString())
          desc = att.getStringValue();
      }

      if (desc == null) desc = "";
    }

    return (desc == null) ? "" : desc;
  }

  /** Set the Unit String for this Variable. Default is to use the "units" attribute. */
  public void setUnitsString( String units) { this.units = units; }

  /** Get the Unit String for the Variable, or null if none. */
  public String getUnitsString() {
    if ((units == null) && (orgVar != null)) {
      Attribute att = orgVar.findAttributeIgnoreCase( "units");
      if ((att != null) && att.isString())
        units = att.getStringValue();
    }
    return units == null ? "" : units;
  }
}

/* Change History:
   $Log: EnhancementsImpl.java,v $
   Revision 1.6  2004/12/07 02:43:21  caron
   *** empty log message ***

   Revision 1.5  2004/12/03 04:46:26  caron
   no message

   Revision 1.4  2004/12/01 05:53:41  caron
   ncml pass 2, new convention parsing

   Revision 1.3  2004/10/19 19:45:02  caron
   misc

   Revision 1.2  2004/09/22 18:43:02  caron
   move common to ucar.unidata; projections use Parameter, no tAttribute

   Revision 1.1  2004/08/16 20:53:48  caron
   2.2 alpha (2)

   Revision 1.3  2004/07/12 23:40:18  caron
   2.2 alpha 1.0 checkin

   Revision 1.2  2004/07/06 19:28:11  caron
   pre-alpha checkin

   Revision 1.1.1.1  2003/12/04 21:05:27  caron
   checkin 2.2
  */
