// $Id: IOServiceProvider.java,v 1.9 2004/10/22 00:52:05 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, strlenwrite to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package ucar.nc2;

import java.io.IOException;

/**
 * This is the service provider interface for the low-level I/O access classes.
 * This is only used by service implementors.
 * @author caron
 * @version 1.0
 */
public interface IOServiceProvider {

   /**
    * Check if this is a valid file for this IOServiceProvider.
    * @param raf RandomAccessFile
    * @return true if valid.
    */ 
  public boolean isValidFile( ucar.unidata.io.RandomAccessFile raf);
  
  /**
   * Open existing file, and populate ncfile with it.
   * @param raf the file to work on, it has already passed the isValidFile() test.
   * @param ncfile add objects to this NetcdfFile
   * @param cancelTask used to monito user cancellation; may be null.
   * @throws IOException
   */
  public void open(ucar.unidata.io.RandomAccessFile raf, ucar.nc2.NetcdfFile ncfile,
                   ucar.nc2.util.CancelTask cancelTask) throws IOException;

  /**
   * Read data from a top level Variable and return a memory resident Array.
   * This Array has the same element type as the Variable, and the requested shape.
   *
   * @param v2 a top-level Variable
   * @param section List of type Range specifying the section of data to read.
   *   There must be a Range for each Dimension in the variable, in order.
   *   Note: no nulls.
   * @return the requested data in a memory-resident Array
   * @throws java.io.IOException
   * @throws ucar.ma2.InvalidRangeException
   * @see ucar.ma2.Range
   */
  public ucar.ma2.Array readData(ucar.nc2.Variable v2, java.util.List section)
         throws java.io.IOException, ucar.ma2.InvalidRangeException;

  /**
   * Read data from a Variable that is nested in one or more Structures. If there are no Structures in the file,
   *   this will never be called.
   *
   * <p>If flatten is false, return nested Arrays of StructureData that correspond to the nested Structures.
   * The innermost Array(s) will match the rank and type of the Variable, but they will be inside Arrays of
   * StructureData.
   *
   * <p>If flatten is true, remove the Arrays of StructureData that wrap the data, and return an Array of the
   * same type as the Variable. The shape of the returned Array will be an accumulation of all the shapes of the
   * Structures containing the variable.
   *
   * <p> v2.getParent() is called to get the containing Structures.
   *
   * @param v2 a nested Variable.
   * @param section List of type Range specifying the section of data to read. There must be a Range for each
   *  Dimension in each parent, as well as in the Variable itself. Must be in order from outer to inner.
   *   Note: no nulls.
   * @param flatten if true, remove containing structures.
   * @return the requested data in a memory-resident Array
   */
  public ucar.ma2.Array readNestedData(ucar.nc2.Variable v2, java.util.List section, boolean flatten)
         throws IOException, ucar.ma2.InvalidRangeException;

  /**
   * Create new file, populate it from the objects in ncfile.
   * @param filename name of file to create.
   * @param ncfile get dimensions, attributes, and variables from here.
   * @param fill if true, write fill value into all variables.
   * @throws IOException
   */
  public void create(String filename, ucar.nc2.NetcdfFile ncfile, boolean fill) throws IOException;

  /**
   * Write data into a variable.
   * @param v2 variable to write; must already exist.
   * @param section List of type Range specifying the section of data to write.
   * There must be a Range for each Dimension in the variable, in order.
   * The shape must match the shape of values.
   * @param values data to write; get shape from here.
   * @throws IOException
   */
  public void writeData(ucar.nc2.Variable v2, java.util.List section, ucar.ma2.Array values)
      throws IOException, ucar.ma2.InvalidRangeException;

  /**
   * Flush all data buffers to disk.
   * @throws IOException
   */
  public void flush() throws IOException;

  /**
   * Close the file.
   * It is the IOServiceProvider's job to close the file (even though it didnt open it),
   * and to free any other resources it has used.
   * @throws IOException
   */
  public void close() throws IOException;

  /**
   * Each Service Provider defines what properties it accepts and what their meaning is.
   * @param name property name
   * @param value property value
   */
  public void setProperty( String name, String value);

  /** Debug info for this object. */
  public String toStringDebug(Object o);


}

/* Change History:
   $Log: IOServiceProvider.java,v $
   Revision 1.9  2004/10/22 00:52:05  caron
   javadoc

   Revision 1.8  2004/09/22 18:44:32  caron
   move common to ucar.unidata

   Revision 1.7  2004/09/22 13:46:35  caron
   *** empty log message ***

   Revision 1.6  2004/08/26 17:55:10  caron
   no message

   Revision 1.5  2004/08/17 19:20:04  caron
   2.2 alpha (2)

   Revision 1.4  2004/08/16 20:53:45  caron
   2.2 alpha (2)

   Revision 1.3  2004/07/12 23:40:17  caron
   2.2 alpha 1.0 checkin

   Revision 1.2  2004/07/06 19:28:10  caron
   pre-alpha checkin

   Revision 1.1.1.1  2003/12/04 21:05:27  caron
   checkin 2.2
 */