// $Id: UI.java,v 1.21 2004/12/07 02:43:20 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package thredds.viewer;

import thredds.ui.*;
import thredds.catalog.*;
import thredds.catalog.ui.*;

import ucar.util.prefs.PreferencesExt;
import ucar.util.prefs.ui.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.event.*;

/**
 * This is the thredds Data Viewer application User Interface.
 *
 * @author John Caron
 * @version $Id: UI.java,v 1.21 2004/12/07 02:43:20 caron Exp $
 */
public class UI extends JPanel {
    // store keys
  static private final String DATASET_LIST = "DatasetList";
  static private final String VIEWER_SIZE = "ViewerSize";
  static private final String SOURCE_WINDOW_SIZE = "SourceWindowSize";
  static private final String XML_WINDOW_SIZE = "XmlWindowSize";

  // tabs
  static private final String TAB_CHOOSER = "ChooserTab";
  static private final String TAB_GRID = "GridTab";
  static private final String TAB_IMAGE = "ImageTab";

  private TopLevel topLevel;
  private PreferencesExt store;
  private Component parent;

  // the main components
  private GridUI gridViewer;
  private ImageUI imageViewer;
  private ThreddsDatasetChooser datasetChooser = null;
  private IndependentWindow sourceWindow = null;
  private ucar.nc2.ui.ToolsUI netcdfUI;

  private thredds.catalog.ui.tools.CatalogEnhancer catEditor = null;
  private thredds.catalog.search.ui.CatalogIndexer catIndexer = null;

  // UI components that need global scope
  private TextGetPutPane sourcePane;
  private JTabbedPane tabbedPane;
  private JMenu debugMenu;

  private TextHistoryPane xmlPane;
  private IndependentDialog xmlWindow = null;

  // the various managers and dialog boxes
  FileManager fileChooser = null; // shared with component viewers

  //private JDialog datasetChooserDialog;
  private IndependentDialog datasetURLDialog = null;

  // actions
  private AbstractAction aboutAction, printAction, sysConfigAction, debugAction, exitAction;
  private AbstractAction /* chooseLocalDatasetAction, chooseThreddsDatasetAction, */ datasetURLAction, saveDatasetAction;
  private AbstractAction clearDebugFlagsAction, setDebugFlagsAction; // clearRecentAction;

  // misc other stuff;
  // private ArrayList datasetList = new ArrayList;
  private int mapBeanCount = 0;
  private int defaultWidth = 700;
  private int defaultHeight = 350;

  // debugging
  private boolean debugBeans = false, debugChooser = false, debugPrint = false, debugHelp = false;
  private boolean debugSelection = false;

  public UI(TopLevel topLevel, PreferencesExt store) {
    this.store = store;
    this.topLevel = topLevel;
    parent = topLevel.getRootPaneContainer().getRootPane();

    enableEvents(AWTEvent.WINDOW_EVENT_MASK);
    Dimension d = (Dimension) store.getBean( VIEWER_SIZE, null);
    setPreferredSize((d != null) ? d : new Dimension(defaultWidth, defaultHeight));

    try  {
      gridViewer = new GridUI((PreferencesExt) store.node("GridUI"), topLevel.getJFrame(), fileChooser, defaultHeight);

      imageViewer = new ImageUI( (PreferencesExt) store.node("ImageUI"), true);
      //stnViewer = new StationUI( topLevel, (PreferencesExt) store.node("StationUI"));

      makeActionsSystem();
      makeActionsDataset();

      makeMenu();
      makeUI();

      //gridViewer.setSelected(false);
      //imageViewer.setSelected(false);
      //stnViewer.setSelected(false);

    } catch (Exception e) {
      System.out.println("UI creation Exception");
      e.printStackTrace();
    }

    // other components
    PreferencesExt fcPrefs = (PreferencesExt) store.node("FileManager");
    FileFilter[] filters = new FileFilter[] {new FileManager.NetcdfExtFilter()};
    fileChooser = new FileManager(null, null, filters, fcPrefs);

    // put it all together into the contentPane
    topLevel.getRootPaneContainer().getContentPane().add(this);
  }

  private void makeUI() throws Exception  {
    tabbedPane = new JTabbedPane(JTabbedPane.TOP);

    /// catalog, DQC, query choosers
    datasetChooser = makeDatasetChooser(tabbedPane); // adds itself to the JTabbedPane

     // panel to show source
    sourcePane = new TextGetPutPane( (PreferencesExt) store.node("getputPane"));
    sourceWindow = new IndependentWindow( "Source", BAMutil.getImage( "GDVs"), sourcePane);
    sourceWindow.setBounds((Rectangle)store.getBean(SOURCE_WINDOW_SIZE, new Rectangle(50, 50, 725, 450)));

     // panel to show xml data
    xmlPane = new TextHistoryPane( false);
    xmlWindow = new IndependentDialog( null, false, "XML data", xmlPane);
    xmlWindow.setBounds((Rectangle)store.getBean(XML_WINDOW_SIZE, new Rectangle(50, 50, 725, 450)));
    // xmlWindow.setIconImage( BAMutil.getImage( "GDVs"));

    /// add viewers
    tabbedPane.addTab("Gridded Data", gridViewer);
    tabbedPane.addTab("Image Data", imageViewer);

    // netcdf tools
    netcdfUI = new ucar.nc2.ui.ToolsUI((PreferencesExt) store.node("netCDFUI"), topLevel.getJFrame());
    tabbedPane.addTab("NetCDF tools", netcdfUI);

    boolean showSystemTools = Debug.isSet("System/showTools");
    if (showSystemTools) {
      catEditor = new thredds.catalog.ui.tools.CatalogEnhancer((PreferencesExt) store.node("catEditor"), topLevel.getJFrame());
      tabbedPane.addTab("Catalog Enhancer", catEditor);

      catIndexer = new thredds.catalog.search.ui.CatalogIndexer((PreferencesExt) store.node("catIndexer"), topLevel.getJFrame());
      tabbedPane.addTab("Indexer", catIndexer);
    }

    setLayout( new BorderLayout());
    add(tabbedPane, BorderLayout.CENTER);
    setTab(TAB_CHOOSER);
  }

  private ThreddsDatasetChooser makeDatasetChooser( JTabbedPane tabbedPane) {

    datasetChooser = new ThreddsDatasetChooser( (PreferencesExt) store.node("ThreddsDatasetChooser"),
                                                tabbedPane);

    if (Debug.isSet("System/filterDataset"))
      datasetChooser.setDatasetFilter( new DatasetFilter.ByServiceType( ServiceType.DODS));

    datasetChooser.addPropertyChangeListener(  new java.beans.PropertyChangeListener() {
      public void propertyChange( java.beans.PropertyChangeEvent e) {
        if (e.getPropertyName().equals("Dataset")) {
          thredds.catalog.InvDataset ds = (thredds.catalog.InvDataset) e.getNewValue();
          setDataset( ds);
        }
      }
    });

    // add a show source button to catalog chooser
    JButton catSource = new JButton("Source");
    catSource.addActionListener( new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        String catURL = datasetChooser.getCatalogChooser().getCurrentURL();
        System.out.println("Catalog Source: url = "+catURL);
        sourcePane.setURL( catURL);
        sourcePane.gotoTop();
        sourceWindow.show();
      }
    });
    datasetChooser.getCatalogChooser().addButton(catSource);

    // add a show source button to dqc chooser
    JButton dcqSource = new JButton("Source");
    dcqSource.addActionListener( new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        String catURL = datasetChooser.getQueryChooser().getCurrentURL();
        String contents = thredds.util.IO.readURLcontents( catURL);
        sourcePane.setURL( catURL);
        sourcePane.gotoTop();
        sourceWindow.show();
      }
    });
    datasetChooser.getQueryChooser().addButton(dcqSource);

    // add a show source button to dqc catalog chooser
    JButton dqcCatSource = new JButton("Source");
    dqcCatSource.addActionListener( new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        String catURL = datasetChooser.getQueryChooser().getCatalogChooser().getCurrentURL();
        System.out.println("DQC Catalog Source: url = "+catURL);
        sourcePane.setURL( catURL);
        sourcePane.gotoTop();
        sourceWindow.show();
      }
    });
    datasetChooser.getQueryChooser().getCatalogChooser().addButton(dqcCatSource);

    return datasetChooser;
  }

     // add a MapBean to the User Interface
  public void addMapBean( thredds.viewer.gis.MapBean mb) {
    gridViewer.addMapBean( mb);
  }

      /** save all data in the PersistentStore */
  public void storePersistentData() {
    store.putBeanObject(VIEWER_SIZE, getSize());
    store.putBeanObject(SOURCE_WINDOW_SIZE, (Rectangle) sourceWindow.getBounds());

    if (fileChooser != null)
      fileChooser.save();

    if (datasetChooser != null)
      datasetChooser.save();

    if (catEditor != null)
      catEditor.save();

    if (catIndexer != null)
      catIndexer.save();

    if (sourcePane != null)
      sourcePane.save();

    gridViewer.storePersistentData();
    imageViewer.save();

    netcdfUI.save();
  }

 /* private boolean chooseDataset(String url) {
    InvDataset invDs = new InvDatasetImpl( fname, ServerType.NETCDF);
    return chooseDataset( invDs);
  } */

  void setDataset(InvDataset ds) {
    if (ds == null) return;

    InvAccess access = ds.getAccess( ServiceType.HTTPServer);
    if (access != null) {
      if ((access.getDataFormatType() == DataFormatType.PLAIN) ||
        (access.getDataFormatType() == DataFormatType.XML)) {
        String urlString = access.getStandardUrlName();
        System.out.println("got station XML data access = "+urlString);
        thredds.util.IO.readURLcontents(urlString);
        xmlPane.setText( thredds.util.IO.readURLcontents(urlString));
        xmlPane.gotoTop();
        xmlWindow.show();
        return;
      }
    }

    if (ds.getDataType() == DataType.IMAGE) {
      setImageAccess( ds);
      setTab(TAB_IMAGE);
      return;
    }

    else { // !! if (ds.getDataType() == DataType.GRID) {
      gridViewer.setDataset( ds);

      setTab(TAB_GRID);
      return;
    }

    /* JOptionPane.showMessageDialog(UI.this, ds.getName()+": unknown data type= "+ds.getDataType(),
        "Error opening dataset", JOptionPane.ERROR_MESSAGE);

    return;  */
  }

  private boolean setImageAccess(InvDataset ds) {
    if (ds == null) return false;

    InvAccess access = ds.getAccess( ServiceType.ADDE);
    if (access == null) return false;

    setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

    if (!(imageViewer.setImageFromUrl( access.getStandardUrlName()))) {
      setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
      return false;
    }

    if (!imageViewer.isSelected()) {
      gridViewer.setSelected(false);
      imageViewer.setSelected(true);
      // stnViewer.setSelected(false);
      //viewerPanel.removeAll();
      //viewerPanel.add(imageViewer, BorderLayout.CENTER);
      revalidate();
    }

    setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    return true;

/*
    } else if (ds.getDataType() == DataType.STATION) {

      if (!(stnViewer.setStations( ds))) {
        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        return false;
      }

      if (!stnViewer.isSelected()) {
        gridViewer.setSelected(false);
        imageViewer.setSelected(false);
        stnViewer.setSelected(true);
        viewerPanel.removeAll();
        viewerPanel.add(stnViewer, BorderLayout.CENTER);
        revalidate();
      }

    }
    return true; */

  }

  /* private void addToDatasetList( String name) {
    int idx;
    if (0 <= (idx = recentDatasetList.indexOf(name))) // move to the top
      recentDatasetList.remove(idx);
    recentDatasetList.add(0, name);
  } */

  // actions that are system-wide
  private void makeActionsSystem() {

    aboutAction = new AbstractAction() {
      public void actionPerformed(ActionEvent evt) {
        new AboutWindow();
      }
    };
    BAMutil.setActionProperties( aboutAction, null, "About", false, 'A', 0);

    /* printAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        PrinterJob printJob = PrinterJob.getPrinterJob();
        PageFormat pf = printJob.defaultPage();

        // do we need to rotate ??
        if (panz.wantRotate( pf.getImageableWidth(), pf.getImageableHeight()))
          pf.setOrientation( PageFormat.LANDSCAPE);
        else
          pf.setOrientation(PageFormat.PORTRAIT);

        printJob.setPrintable(controller.getPrintable(), pf);
        if (printJob.printDialog()) {
          try {
            if (Debug.isSet("print.job")) System.out.println("call printJob.print");
            printJob.print();
            if (Debug.isSet("print.job")) System.out.println(" printJob done");
          } catch (Exception PrintException) {
            PrintException.printStackTrace();
          }
        }
      }
    };
    BAMutil.setActionProperties( printAction, "Print", "Print...", false, 'P', KeyEvent.VK_P);

    sysConfigAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if (sysConfigDialog == null)
          makeSysConfigWindow();
        sysConfigDialog.show();
      }
    };
    BAMutil.setActionProperties( sysConfigAction, "Preferences", "Configure...", false, 'C', -1);

    clearDebugFlagsAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) { Debug.clear(); }
    };
    BAMutil.setActionProperties( clearDebugFlagsAction, null, "Clear DebugFlags", false, 'D', -1);

    clearRecentAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        recentDatasetList = new ArrayList();
      }
    };
    BAMutil.setActionProperties( clearRecentAction, null, "Clear Recent Datasets", false, 'R', -1);
    */

    clearDebugFlagsAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) { /* Debug.clear(); */ }
    };
    BAMutil.setActionProperties( clearDebugFlagsAction, null, "Clear Debug Flags", false, 'D', -1);

    setDebugFlagsAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        // LOOK set netcdf debug flags
        
        thredds.catalog.InvCatalogFactory.debugURL =  Debug.isSet("InvCatalogFactory/debugURL");
        thredds.catalog.InvCatalogFactory.debugOpen =  Debug.isSet("InvCatalogFactory/debugOpen");
        thredds.catalog.InvCatalogFactory.debugVersion =  Debug.isSet("InvCatalogFactory/debugVersion");
        thredds.catalog.InvCatalogFactory.showParsedXML =  Debug.isSet("InvCatalogFactory/showParsedXML");
        thredds.catalog.InvCatalogFactory.showStackTrace =  Debug.isSet("InvCatalogFactory/showStackTrace");
        thredds.catalog.InvCatalogFactory.debugXML =  Debug.isSet("InvCatalogFactory/debugXML");
        thredds.catalog.InvCatalogFactory.debugDBurl =  Debug.isSet("InvCatalogFactory/debugDBurl");
        thredds.catalog.InvCatalogFactory.debugXMLopen =  Debug.isSet("InvCatalogFactory/debugXMLopen");
        thredds.catalog.InvCatalogFactory.showCatalogXML =  Debug.isSet("InvCatalogFactory/showCatalogXML");
      }
    };
    BAMutil.setActionProperties( setDebugFlagsAction, null, "Set Debug Flags", false, 'S', -1);

    exitAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        topLevel.close();
      }
    };
    BAMutil.setActionProperties( exitAction, "Exit", "Exit", false, 'X', -1);

  }

  // actions that control the dataset
  private void makeActionsDataset() {

      /* choose local dataset
    chooseLocalDatasetAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if ( null == fileChooser) {
          String dirName = store.get(FILECHOOSER_DEFAULTDIR, null);
          fileChooser = new FileManager(parent, dirName, new FileManager.NetcdfExtFilter());
        }
        java.io.File file = fileChooser.chooseFile();
        if (file == null) return;

        InvDataset invDs;
        try {
          invDs = new InvDatasetImpl( file.toURI().toString(), DataType.GRID, ServiceType.NETCDF);
        } catch (Exception ue) {
          javax.swing.JOptionPane.showMessageDialog(UI.this, "Invalid filename = <"+file+">\n"+ue.getMessage());
          ue.printStackTrace();
          return;
        }
        setDataset( invDs);
      }
    };
    BAMutil.setActionProperties( chooseLocalDatasetAction, "Choose", "open Local dataset...", false, 'L', -1); */

    /*choose THREDDS dataset
    chooseThreddsDatasetAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        datasetChooserDialog.show();
      }
    };
    BAMutil.setActionProperties( chooseThreddsDatasetAction, "Choose", "open THREDDS dataset...", false, 'R', -1);
    */
      /* enter dataset URL
    datasetURLAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        if (datasetURLDialog == null) {
          datasetURLpp = new faqo.ui.PreferencePanel( "Enter Dataset URL", new faqo.ui.PreferenceData(store), null);
          datasetURLpp.addTextField( DATASET_URL, "dataset URL", "                                                  ");
          datasetURLpp.finishSetup();
          datasetURLpp.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              String url = (String) datasetURLpp.getValue(DATASET_URL);
              System.out.println("dataset url = "+url);
              if (chooseDataset( url)) {
                datasetURLDialog.setVisible(false);
              }
            }
          });
          datasetURLDialog = new IndependentWindow( "Enter Dataset URL", datasetURLpp);
        }
        datasetURLDialog.show();
      }
    };
    BAMutil.setActionProperties( datasetURLAction, null, "enter dataset URL...", false, 'U', -1);

    /* saveDatasetAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        String fname = controller.getDatasetName();
        if (fname != null) {
          savedDatasetList.add( fname);
          BAMutil.addActionToMenu( savedDatasetMenu, new DatasetAction( fname), 0);
        }
      }
    };
    BAMutil.setActionProperties( saveDatasetAction, null, "save dataset", false, 'S', 0);
    */

  }

  private void setTab( String tabName) {
    if (tabName.equals(TAB_CHOOSER))
      tabbedPane.setSelectedIndex(0);
    else if (tabName.equals(TAB_GRID))
      tabbedPane.setSelectedComponent(gridViewer);
    else if (tabName.equals(TAB_IMAGE))
      tabbedPane.setSelectedComponent(imageViewer);
  }

  private void makeMenu() {
    JMenuBar mb = new JMenuBar();
    JRootPane rootPane = topLevel.getRootPaneContainer().getRootPane();
    rootPane.setJMenuBar(mb);

    /// System menu
    JMenu sysMenu = new JMenu("System");
    sysMenu.setMnemonic('S');
    mb.add(sysMenu);
    //BAMutil.addActionToMenu( sysMenu, printAction);

    JMenu plafMenu = new JMenu("Look and Feel");
    plafMenu.setMnemonic('L');
    sysMenu.add(plafMenu);
    PLAF plaf = new PLAF(rootPane);
    plaf.addToMenu( plafMenu);

    if (Debug.isSet("util/configure")) BAMutil.addActionToMenu( sysMenu, sysConfigAction);

    debugMenu = (JMenu) sysMenu.add(new JMenu("Debug"));
    debugMenu.addMenuListener( new MenuListener() {
      public void menuSelected(MenuEvent e) { ucar.util.prefs.ui.Debug.constructMenu(debugMenu); }
      public void menuDeselected(MenuEvent e) {}
      public void menuCanceled(MenuEvent e) {}
    });

    BAMutil.addActionToMenu( sysMenu, setDebugFlagsAction);
    //BAMutil.addActionToMenu( sysMenu, clearDebugFlagsAction);
    /* if (null != System.getProperty("admin")) {
      JMenu adminMenu = new JMenu("Admin");
      plafMenu.setMnemonic('A');
      sysMenu.add(adminMenu);
      BAMutil.addActionToMenu( adminMenu, clearRecentAction);
      BAMutil.addActionToMenu( adminMenu, clearDebugFlagsAction);
    } */

    BAMutil.addActionToMenu( sysMenu, exitAction);
    if (topLevel.isApplet())
      exitAction.setEnabled( false);

    /* Data menu
    dataMenu = new JMenu("Dataset");
    dataMenu.setMnemonic('D');
    mb.add(dataMenu);
    //BAMutil.addActionToMenu( dataMenu, chooseLocalDatasetAction);
    //BAMutil.addActionToMenu( dataMenu, chooseThreddsDatasetAction);
    //BAMutil.addActionToMenu( dataMenu, datasetURLAction);

      /* permanent datasets
      savedDatasetMenu = new JMenu("saved");
      //dataMenu.add( savedDatasetMenu);
      savedDatasetList = (ArrayList) store.getBean( "savedDatasetList", null);
      if (null != savedDatasetList) {
        Iterator iter = savedDatasetList.iterator();
        while (iter.hasNext())
          BAMutil.addActionToMenu( savedDatasetMenu, new DatasetAction( (String) iter.next()));
      } else
        savedDatasetList = new ArrayList(); */

        // Info
    /* JMenu infoMenu = new JMenu("Info");
    infoMenu.setMnemonic('I');
    mb.add(infoMenu);

    /// Configure
    configMenu = new JMenu("Configure");
    configMenu.setMnemonic('C');
    mb.add(configMenu);

    //// tools menu
    JMenu toolMenu = new JMenu("Tools");
    toolMenu.setMnemonic( 'T');
    mb.add(toolMenu); */

   // Help
    JMenu helpMenu = new JMenu("Help");
    helpMenu.setMnemonic('H');
    mb.add(helpMenu);
    // BAMutil.addActionToMenu( helpMenu, controller.helpAction);
    BAMutil.addActionToMenu( helpMenu, aboutAction);

    // other UI's add their actions to menu
    //gridViewer.addActionsToMenus(dataMenu, configMenu, toolMenu);
  }

    //////////////////
  private class AboutWindow extends javax.swing.JWindow  {
    public AboutWindow() {
      super();

      JLabel lab1 = new JLabel(
        "<html> <body bgcolor=\"#FFECEC\"> <center>"+
        "<h1>THREDDS Toolset</h1>" +
        "<b>"+getVersion()+"</b>" +
        "<br><i>http://www.unidata.ucar.edu/projects/THREDDS/</i><br>" +
        "</center><br>With thanks to these <b>Open Source</b> contributers:" +
        "<ul>" +
        "<li><b>VisAD/ADDE</b>: Bill Hibbert, Don Murray, Tom Whittaker</li>" +
        "<li><b>Java-DODS</b>: Nathan Potter, James Gallagher, Don Denbo, et. al.(http://www.unidata.ucar.edu/packages/dods/index.html)</li>" +
        "<li><b>JDOM</b> library: Jason Hunter, Brett McLaughlin et al (www.jdom.org)</li>" +
        "<li><b>JUnit</b> library: Erich Gamma, Kent Beck, Erik Meade, et al (http://sourceforge.net/projects/junit/)</li>" +
        "<li><b>Log4J</b> library: (http://logging.apache.org/log4j/) </li>" +
        "<li><b>HTTPClient</b> library: Ronald Tschalr (http://www.innovation.ch/java/HTTPClient/)</li>" +
       " </ul><center>Special thanks to <b>Sun Microsystems</b> (java.sun.com) for the platform on which we stand."+
       " </center></body></html> "
      );


      JPanel main = new JPanel( new BorderLayout());
      main.setBorder( new javax.swing.border.LineBorder( Color.BLACK));

      JLabel gdvIcon = new JLabel( new ImageIcon(BAMutil.getImage( "GDV")));
      gdvIcon.setBackground(new Color(0xFFECEC));
      gdvIcon.setOpaque(true);

      JLabel threddsLogo = new JLabel( Resource.getIcon( "/resources/icons/THREDDSlogo.jpg", false));
      threddsLogo.setBackground(new Color(0xFFECEC));
      threddsLogo.setOpaque(true);

      main.add(threddsLogo, BorderLayout.NORTH);
      main.add(lab1, BorderLayout.CENTER);
      main.add(gdvIcon, BorderLayout.SOUTH);
      getContentPane().add(main);
      pack();

      Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
      Dimension labelSize = this.getPreferredSize();
      setLocation(screenSize.width/2 - (labelSize.width/2),
                  screenSize.height/2 - (labelSize.height/2));
      addMouseListener(new MouseAdapter() {
        public void mousePressed(MouseEvent e) {
            setVisible(false);
            dispose();
        }
      });
      setVisible(true);
    }


      private String version = null;
      private String getVersion() {

        try {
          InputStream is = thredds.util.Resource.getFileResource( "/README");
             // DataInputStream dataIS = new DataInputStream( new BufferedInputStream(ios, 20000));
          BufferedReader dataIS = new BufferedReader(new InputStreamReader(is));
          StringBuffer sbuff = new StringBuffer();
          for (int i=0; i<3; i++) {
            sbuff.append( dataIS.readLine());
            sbuff.append( "<br>");
          }
          version = sbuff.toString();
        } catch( IOException ioe) {
          ioe.printStackTrace();
          version = "version unknown";
        }

        return version;
      }

  }



}

/* Change History:
   $Log: UI.java,v $
   Revision 1.21  2004/12/07 02:43:20  caron
   *** empty log message ***

   Revision 1.20  2004/12/01 05:54:24  caron
   improve FileChooser

   Revision 1.19  2004/11/07 03:00:48  caron
   *** empty log message ***

   Revision 1.18  2004/11/04 20:16:43  caron
   no message

   Revision 1.17  2004/09/30 00:33:39  caron
   *** empty log message ***

   Revision 1.16  2004/09/25 00:09:43  caron
   add images, thredds tab

   Revision 1.15  2004/09/24 03:26:37  caron
   merge nj22

   Revision 1.14  2004/06/19 00:46:39  caron
   funny null  syntax

   Revision 1.13  2004/06/18 21:54:27  caron
   update dqc 0.3

   Revision 1.12  2004/06/12 02:12:39  caron
   update viewer

   Revision 1.11  2004/05/21 05:57:35  caron
   release 2.0b

   Revision 1.10  2004/05/11 23:30:37  caron
   release 2.0a

   Revision 1.9  2004/03/19 20:18:02  caron
   use thredds.datamodel to connect catalog with data types

   Revision 1.8  2004/03/05 23:45:43  caron
   1.3.1

   Revision 1.7  2004/02/21 02:18:50  caron
   add java.net.Authenticator.setDefault()

   Revision 1.6  2004/02/20 05:02:55  caron
   release 1.3

   Revision 1.5  2003/12/04 22:27:48  caron
   *** empty log message ***

   Revision 1.4  2003/05/29 23:07:52  john
   bug fixes

   Revision 1.3  2003/03/17 21:12:35  john
   new viewer

   Revision 1.2  2003/01/07 14:57:20  john
   lib updates

   Revision 1.1  2002/12/13 00:51:11  caron
   pass 2

   Revision 1.3  2002/10/18 18:21:18  caron
   thredds server

   Revision 1.2  2002/04/29 22:39:23  caron
   add StationUI, clean up

   Revision 1.1.1.1  2002/02/26 17:24:48  caron
   import sources

*/


