// $Id: TimeDuration.java,v 1.3 2004/09/24 03:26:32 caron Exp $
/*
 * Copyright 2002-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package thredds.datatype;

import ucar.nc2.units.*;

/**
 * Implements the thredds "duration" XML element type.
 *
 * A duration can be one of the following:
 <ol>
   <li> a valid udunits string compatible with "secs"
   <li> NOT DONE YET an xsd:duration type specified in the following form "PnYnMnDTnHnMnS" where:
    <ul>
    <li>P indicates the period (required)
    <li>nY indicates the number of years
    <li>nM indicates the number of months
    <li>nD indicates the number of days
    <li>T indicates the start of a time section (required if you are going to specify hours, minutes, or seconds)
    <li>nH indicates the number of hours
    <li>nM indicates the number of minutes
    <li>nS indicates the number of seconds
  </ul>
 </ol>
 *
 * @see "http://www.unidata.ucar.edu/projects/THREDDS/tech/catalog/InvCatalogSpec.html#durationType"
 * @author john caron
 * @version $Revision: 1.3 $ $Date: 2004/09/24 03:26:32 $
 */


public class TimeDuration {
  private String text;
  private TimeUnit timeUnit;
  private boolean isBlank;

  /** copy constructor */
  public TimeDuration( TimeDuration src) {
    text = src.getText();
    timeUnit = src.getTimeUnit();
    isBlank = src.isBlank();
  }

  public TimeDuration( TimeUnit timeUnit) {
    this.timeUnit = timeUnit;
    this.text = timeUnit.toString();
  }

  public TimeDuration(String text) throws java.text.ParseException {
    text = (text == null) ? "" : text.trim();
    this.text = text;

    // see if its blank
    if (text.length() == 0) {
      isBlank = true;
      return;
    }

    // LOOK need xsd:duration parsing

    // see if its a udunits string
    try {
      timeUnit = new TimeUnit(text);
    }
    catch (Exception e) {
      throw new java.text.ParseException(e.getMessage(), 0);
    }

  }

  // get the duration in seconds
  public double getSeconds() {
    return timeUnit.getSeconds();
  }

  public boolean isBlank() {
    return isBlank;
  }

  public TimeUnit getTimeUnit() {
    return timeUnit;
  }

  public String getText() {
    return text;
  }

  public String toString() {
    return text;
  }

  public int hashCode() {
    if (isBlank())
      return 0;
    return timeUnit.hashCode();
  }

  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (! (o instanceof DateType))
      return false;
    return o.hashCode() == this.hashCode();
  }

  ////////////////////////////////////////////
  // test

  private static void doDuration(String s) {
    try {
      System.out.println("start = (" + s + ")");
      TimeDuration d = new TimeDuration(s);
      System.out.println("duration = (" + d.toString() + ")");
    }
    catch (java.text.ParseException e) {
      e.printStackTrace();
    }
  }

  public static void main(String[] args) {

    doDuration("3 days");
  }
}

/* Change History:
   $Log: TimeDuration.java,v $
   Revision 1.3  2004/09/24 03:26:32  caron
   merge nj22

*/