// $Id: DirectoryScanner.java,v 1.2 2004/12/15 17:51:03 edavis Exp $
package thredds.cataloggen;

import org.apache.log4j.Logger;
import thredds.cataloggen.config.DatasetSource;
import thredds.cataloggen.config.DatasetSourceType;
import thredds.cataloggen.config.DatasetSourceStructure;
import thredds.cataloggen.config.ResultService;
import thredds.catalog.*;

import java.io.File;
import java.io.IOException;

/**
 * Allow generation of THREDDS InvCatalog documents from the contents of a local directory.
 *
 * @author edavis
 * @since 2004-12-10T15:33:32-0700
 */
public class DirectoryScanner
{
  private static Logger logger = Logger.getLogger( DirectoryScanner.class.getName() );

  private String serviceId;
  private String serviceTitle;
  private ServiceType serviceType;
  private String serviceBaseUrl;
  private File serviceRootDir;

  private ResultService service;

  // String name,
  // DatasetSourceStructure structure,
  // String accessPoint,
  // ResultService resultService
  //   String name, String serviceTypeName,
  //   String base, String suffix(???),
  //   String accessPointHeader

  /**
   * Construct a DirectoryScanner given information about the data server to be cataloged.
   *
   * @param serviceId - a String used to identify the service.
   * @param serviceTitle - a title for the service (used as the title of the top-level dataset).
   * @param serviceType - a THREDDS InvCatalog valid service type.
   * @param serviceBaseUrl - the base URL for accessing the service.
   * @param serviceRootDir - the local directory that the service uses as its docRoot.
   *
   * @throws IllegalArgumentException if invalid service type is given or root directory is not a directory.
   */
  public DirectoryScanner( String serviceId, String serviceTitle, ServiceType serviceType,
                           String serviceBaseUrl, File serviceRootDir)
  {
    this.serviceId = serviceId;
    this.serviceTitle = serviceTitle;
    this.serviceType = serviceType;
    if ( this.serviceType == null) throw new IllegalArgumentException( "Null service type not allowed.");
    this.serviceBaseUrl = serviceBaseUrl;
    this.serviceRootDir = serviceRootDir;
    if ( ! this.serviceRootDir.isDirectory()) throw new IllegalArgumentException( "Root directory is not a directory <" + serviceRootDir.getAbsolutePath() + ">.");
    this.service = new ResultService( this.serviceId, this.serviceType,
                                      this.serviceBaseUrl, null, this.serviceRootDir.getAbsolutePath() );
  }

  /**
   * Return a catalog for the given directory.
   *
   * @param directory - the directory to represent as a catalog.
   * @return An InvCatalog that represents the given directory of datasets.
   */
  public InvCatalog getDirCatalog( File directory )
  {
    if ( ! directory.isDirectory()) throw new IllegalArgumentException( "Given directory is not a directory <" + directory.getAbsolutePath() + ">.");
    DatasetSource dsSource = DatasetSource.newDatasetSource( null, null, directory.getName(),
                                                             DatasetSourceType.LOCAL, DatasetSourceStructure.DIRECTORY_TREE,
                                                             directory.getAbsolutePath(), this.service);
    InvDataset topDs = dsSource.expand();
    // @todo Use DatasetNamer here instead of doing it by hand.
    if( topDs.getName().equals( ""))
      ( (InvDatasetImpl) topDs).setName( this.serviceTitle);

    return( topDs.getParentCatalog());
  }

  /**
   * Return a string representation of a catalog for the given directory.
   *
   * @param directory - the directory to catalog.
   * @return The string representation of the catalog.
   */
  public String getDirCatalogAsString( File directory )
  {
    String results = null;
    InvCatalogFactory catFactory = new InvCatalogFactory( "default", true );
    // Write the catalog to standard output.
    try
    {
      results = catFactory.writeXML( (InvCatalogImpl) this.getDirCatalog( directory));
    }
    catch ( IOException e )
    {
      logger.warn( "IOException while writing catalog to string: " + e.getMessage() );
      return( null);
    }

    return( results);
  }
}

/*
 * $Log: DirectoryScanner.java,v $
 * Revision 1.2  2004/12/15 17:51:03  edavis
 * Changes to clean up ResultService. Changes to add a server title to DirectoryScanner (becomes the title of the top-level dataset).
 *
 * Revision 1.1  2004/12/14 22:47:22  edavis
 * Add simple interface to thredds.cataloggen and continue adding catalogRef capabilities.
 *
 */