// $Id: DCWriter.java,v 1.3 2004/09/24 03:26:28 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package thredds.catalog.dl;

import thredds.catalog.*;
import thredds.datatype.DateType;


import org.jdom.*;
import org.jdom.output.*;

import java.io.*;
import java.util.*;

public class DCWriter {
  private static final Namespace defNS = Namespace.getNamespace("http://purl.org/dc/elements/1.1/");
  private static final Namespace xsiNS = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");
  private static final String schemaLocation = defNS.getURI() + " http://www.unidata.ucar.edu/schemas/other/dc/dc.xsd";
  private static final String schemaLocationLocal = defNS.getURI() + " file://P:/schemas/other/dc/dc.xsd";
  private static String threddsServerURL = "http://localhost:8080/thredds/subset.html";

  private InvCatalog cat;
  public DCWriter( InvCatalog cat) {
    this.cat = cat;
  }

  public void writeItems( String fileDir) {
    File dir = new File(fileDir);
    if (!dir.exists())
      dir.mkdirs();

    List datasets = cat.getDatasets();
    for (int i=0; i<datasets.size(); i++) {
      InvDataset elem = (InvDataset) datasets.get(i);
      doDataset( elem, fileDir);
    }

  }

  private void doDataset( InvDataset ds, String fileDir) {
    if (ds.isHarvest() && (ds.getID() != null)) {
      String fileOutName = fileDir+"/"+ds.getID()+".dc.xml";
      try {
        OutputStream out = new BufferedOutputStream(new FileOutputStream(
            fileOutName));
        writeOneItem(ds, System.out);
        writeOneItem(ds, out);
        out.close();
        return;
      } catch (IOException ioe) {
        ioe.printStackTrace();
      }
    }

    List datasets = ds.getDatasets();
    for (int i=0; i<datasets.size(); i++) {
      InvDataset nested = (InvDataset) datasets.get(i);
      doDataset( nested, fileDir);
    }
  }

  private void writeOneItem( InvDataset ds, OutputStream out) throws IOException {
    Element rootElem = new Element("dc", defNS);
    Document doc = new Document(rootElem);
    writeDataset( ds, rootElem);
    rootElem.addNamespaceDeclaration(xsiNS);
    rootElem.setAttribute("schemaLocation", schemaLocation, xsiNS);

    // Output the document, use standard formatter
    XMLOutputter fmt = new XMLOutputter("  ", true);
    fmt.output( doc, out);
  }

  /*
        <xs:element ref="dc:title"/>
        <xs:element ref="dc:creator"/>
        <xs:element ref="dc:subject"/>
        <xs:element ref="dc:description"/>
        <xs:element ref="dc:publisher"/>
        <xs:element ref="dc:contributor"/>
        <xs:element ref="dc:date"/>
        <xs:element ref="dc:type"/>
        <xs:element ref="dc:format"/>
        <xs:element ref="dc:identifier"/>
        <xs:element ref="dc:source"/>
        <xs:element ref="dc:language"/>
        <xs:element ref="dc:relation"/>
        <xs:element ref="dc:coverage"/>
        <xs:element ref="dc:rights"/>
        <!-- controlled vocabulary from dcmitype -->
        <xs:element ref="dcmitype:dcmitype"/>
  */
  public void writeDataset(InvDataset ds, Element rootElem) {

    rootElem.addContent( new Element("title", defNS).addContent(ds.getName()));

    rootElem.addContent( new Element("Entry_ID", defNS).addContent(ds.getUniqueID()));

    // keywords
    List list = ds.getKeywords();
    if (list.size() > 0) {
      for (int i=0; i<list.size(); i++) {
        ThreddsMetadata.Vocab k = (ThreddsMetadata.Vocab) list.get(i);
        rootElem.addContent( new Element("Keyword", defNS).addContent( k.getText()));
      }
    }

    //temporal
    ThreddsMetadata.TimeCoverage tm = ds.getTimeCoverage();
    Element tmElem = new Element("Temporal_Coverage", defNS);
    rootElem.addContent( tmElem);

    tmElem.addContent( new Element("Start_Date", defNS).addContent( tm.getStart().toDateString()));
    tmElem.addContent( new Element("End_Date", defNS).addContent( tm.getEnd().toDateString()));

    //geospatial
    ThreddsMetadata.GeospatialCoverage geo = ds.getGeospatialCoverage();
    Element geoElem = new Element("Spatial_Coverage", defNS);
    rootElem.addContent( geoElem);

    geoElem.addContent( new Element("Southernmost_Latitude", defNS).addContent( Double.toString(geo.getLatSouth())));
    geoElem.addContent( new Element("Northernmost_Latitude", defNS).addContent(Double.toString(geo.getLatNorth())));
    geoElem.addContent( new Element("Westernmost_Latitude", defNS).addContent(Double.toString(geo.getLonWest())));
    geoElem.addContent( new Element("Easternmost_Latitude", defNS).addContent(Double.toString(geo.getLonEast())));

    rootElem.addContent( new Element("Use_Constraints", defNS).addContent(ds.getDocumentation("rights")));

    // data center
    list = ds.getPublishers();
    if (list.size() > 0) {
      for (int i=0; i<list.size(); i++) {
        ThreddsMetadata.Source p = (ThreddsMetadata.Source) list.get(i);
        Element dataCenter = new Element("Data_Center", defNS);
        rootElem.addContent( dataCenter);
        writePublisher(p, dataCenter);
      }
    }

    rootElem.addContent( new Element("Summary", defNS).addContent(ds.getDocumentation("summary")));

    Element primaryURLelem = new Element("Related_URL", defNS);
    rootElem.addContent( primaryURLelem);

    String primaryURL = threddsServerURL +
        "?catalog="+((InvCatalogImpl)ds.getParentCatalog()).getBaseURI().toString() +
        "&dataset="+ds.getID();
    primaryURLelem.addContent( new Element("URL_Content_Type", defNS).addContent("THREDDS access page"));
    primaryURLelem.addContent( new Element("URL", defNS).addContent(primaryURL));

    DateType today = new DateType(false, new Date());
    rootElem.addContent(new Element("DIF_Creation_Date", defNS).addContent(today.toDateString()));

  }

  protected void writePublisher(ThreddsMetadata.Source p, Element dataCenter) {
    Element name = new Element("Data_Center_Name", defNS);
    dataCenter.addContent( name);
    name.addContent( new Element("Short_Name", defNS).addContent(p.getName()));
    //name.addContent( new Element("Long_Name", defNS).addContent(p.getLongName()));

    if ((p.getUrl() != null) && p.getUrl().length() > 0)
      dataCenter.addContent( new Element("Data_Center_URL", defNS).addContent(p.getUrl()));

    Element person = new Element("Personnel", defNS);
    dataCenter.addContent( person);
    person.addContent( new Element("Role", defNS).addContent("DATA CENTER CONTACT"));
    person.addContent( new Element("Email", defNS).addContent(p.getEmail()));
  }

  private void writeVariable( Element param, ThreddsMetadata.Variable v) {
    String vname = v.getVocabularyName();
    StringTokenizer stoker = new StringTokenizer(vname,">");
    if (stoker.hasMoreTokens())
      param.addContent( new Element("Category", defNS).addContent(stoker.nextToken().trim()));
    if (stoker.hasMoreTokens())
      param.addContent( new Element("Topic", defNS).addContent(stoker.nextToken().trim()));
    if (stoker.hasMoreTokens())
      param.addContent( new Element("Term", defNS).addContent(stoker.nextToken().trim()));
    if (stoker.hasMoreTokens())
      param.addContent( new Element("Variable", defNS).addContent(stoker.nextToken().trim()));
    if (stoker.hasMoreTokens())
      param.addContent( new Element("Detailed_Variable", defNS).addContent(stoker.nextToken().trim()));
  }

  // test
  private static void doOne( InvCatalogFactory fac, String url) {
    System.out.println("***read "+url);
    try {
        InvCatalogImpl cat = (InvCatalogImpl) fac.readXML(url);
        StringBuffer buff = new StringBuffer();
        boolean isValid = cat.check( buff, false);
        System.out.println("catalog <" + cat.getName()+ "> "+ (isValid ? "is" : "is not") + " valid");
        System.out.println(" validation output=\n" + buff);
        // System.out.println(" catalog=\n" + fac.writeXML(cat));

        DCWriter w = new DCWriter( cat);
        w.writeItems( "C:/temp/dif");
      } catch (Exception e) {
        e.printStackTrace();
      }

  }

   /** testing */
  public static void main (String[] args) throws Exception {
    InvCatalogFactory catFactory = InvCatalogFactory.getDefaultFactory(true);

    doOne(catFactory, "file:///C:/dev/thredds/catalog/test/data/TestHarvest.xml");
  }

}

/* Change History:
   $Log: DCWriter.java,v $
   Revision 1.3  2004/09/24 03:26:28  caron
   merge nj22

   Revision 1.2  2004/06/09 00:27:26  caron
   version 2.0a release; cleanup javadoc

 */