// $Id: ADNWriter.java,v 1.3 2004/09/24 03:26:28 caron Exp $
/*
 * Copyright 1997-2004 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package thredds.catalog.dl;

import thredds.catalog.*;
import thredds.datatype.DateType;


import org.jdom.*;
import org.jdom.output.*;

import java.io.*;
import java.util.*;

public class ADNWriter {
  private static final Namespace defNS = Namespace.getNamespace("http://adn.dlese.org");
  private static final Namespace xsiNS = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");
  private static final String schemaLocation = defNS.getURI() + " http://www.dlese.org/Metadata/adn-item/0.6.50/record.xsd";
  private static final String schemaLocationLocal = defNS.getURI() + " file://c:/dev/metadata/adn/0.6.50/record.xsd";
  private static String threddsServerURL = "http://localhost:8080/thredds/subset.html";

  public ADNWriter( ) {
  }

  public void writeDatasetEntries( InvCatalog cat, String fileDir, StringBuffer mess) {
    File dir = new File(fileDir);
    if (!dir.exists())
      dir.mkdirs();

    List datasets = cat.getDatasets();
    for (int i=0; i<datasets.size(); i++) {
      InvDataset elem = (InvDataset) datasets.get(i);
      doDataset( elem, fileDir, mess);
    }

  }

  private void doDataset( InvDataset ds, String fileDir, StringBuffer mess) {
    if (!ds.isHarvest()) {
      mess.append( "Dataset "+ ds.getName()+ " id = "+ds.getID()+" has harvest = false\n");

    } else if (isDatasetUseable( ds, mess)) {
      String fileOutName = fileDir+"/"+ds.getID()+".xml";
      try {
        OutputStream out = new BufferedOutputStream(new FileOutputStream( fileOutName));
        writeOneItem(ds, out);
        out.close();
        return;
      } catch (IOException ioe) {
        ioe.printStackTrace();
      }
    }

    List datasets = ds.getDatasets();
    for (int i=0; i<datasets.size(); i++) {
      InvDataset nested = (InvDataset) datasets.get(i);
      doDataset( nested, fileDir, mess);
    }

  }

  public boolean isDatasetUseable(InvDataset ds, StringBuffer sbuff) {
    boolean ok = true;
    sbuff.append("Dataset "+ ds.getName()+ " id = "+ds.getID()+": ");

    if (ds.getName() == null) {
      ok = false;
      sbuff.append(" missing Name field\n");
    }

    if (ds.getUniqueID() == null) {
      ok = false;
      sbuff.append(" missing ID field\n");
    }

    List list = ds.getPublishers();
    if ((list == null) || (list.size() == 0)) {
      ok = false;
      sbuff.append(" must have publisher element that defines the data center\n");
    }

    String summary = ds.getDocumentation("summary");
    if (summary == null) {
      ok = false;
      sbuff.append(" must have documentation element of type summary\n");
    }

    String rights = ds.getDocumentation("rights");
    if (rights == null) {
      ok = false;
      sbuff.append(" must have documentation element of type rights\n");
    }

    sbuff.append(" useable= "+ok+"\n");
    return ok;
  }



  private void writeOneItem( InvDataset ds, OutputStream out) throws IOException {
    Element rootElem = new Element("itemRecord", defNS);
    Document doc = new Document(rootElem);
    writeDataset( ds, rootElem);
    rootElem.addNamespaceDeclaration(xsiNS);
    rootElem.setAttribute("schemaLocation", schemaLocationLocal, xsiNS);

    // Output the document, use standard formatter
    XMLOutputter fmt = new XMLOutputter("  ", true);
    fmt.output( doc, out);
  }

  public void writeDataset(InvDataset ds, Element rootElem) {
    // general
    Element generalElem = new Element("general", defNS);
    rootElem.addContent( generalElem);

    generalElem.addContent( new Element("title", defNS).addContent(ds.getName()));
    generalElem.addContent( new Element("description", defNS).addContent(ds.getDocumentation("summary")));
    generalElem.addContent( new Element("language", defNS).addContent("en"));

    Element subjects = new Element("subjects", defNS);
    generalElem.addContent( subjects);
    subjects.addContent( new Element("subject", defNS).addContent("DLESE:Atmospheric science"));

    List list = ds.getKeywords();
    if (list.size() > 0) {
      Element keywords = new Element("keywords", defNS);
      generalElem.addContent( keywords);
      for (int i=0; i<list.size(); i++) {
        ThreddsMetadata.Vocab k = (ThreddsMetadata.Vocab) list.get(i);
        keywords.addContent( new Element("keyword", defNS).addContent( k.getText()));
      }
    }

    // life cycle
    Element lifeElem = new Element("lifecycle", defNS);
    rootElem.addContent( lifeElem);

    list = ds.getPublishers();
    if (list.size() > 0) {
      Element contributors = new Element("contributors", defNS);
      lifeElem.addContent( contributors);
      for (int i=0; i<list.size(); i++) {
        ThreddsMetadata.Source p = (ThreddsMetadata.Source) list.get(i);
        contributors.addContent( writePublisher(p));
      }
    }

    // metaMetadata
    Element metaElem = new Element("metaMetadata", defNS);
    rootElem.addContent( metaElem);

    Element entries = new Element("catalogEntries", defNS);
    metaElem.addContent( entries);

    Element entry = new Element("catalog", defNS);
    entries.addContent( entry);
    entry.setAttribute("entry", ds.getUniqueID());
    entry.addContent("THREDDS-motherlode");

    DateType today = new DateType(false, new Date());
    Element dateInfo = new Element("dateInfo", defNS);
    metaElem.addContent( dateInfo);
    dateInfo.setAttribute("created", today.toDateString());
    dateInfo.setAttribute("accessioned", today.toDateString());

    Element status = new Element("statusOf", defNS);
    metaElem.addContent( status);
    status.setAttribute("status", "Accessioned");

    metaElem.addContent( new Element("language", defNS).addContent("en"));
    metaElem.addContent( new Element("scheme", defNS).addContent("ADN (ADEPT/DLESE/NASA Alexandria Digital Earth Prototype/Digital Library for Earth System Education/National Aeronautics and Space Administration)"));
    metaElem.addContent( new Element("copyright", defNS).addContent("Public Domain"));

    Element terms = new Element("termsOfUse", defNS);
    metaElem.addContent( terms);
    terms.addContent( "Public Domain");

    // technical
    Element technical = new Element("technical", defNS);
    rootElem.addContent( technical);

    Element online = new Element("online", defNS);
    technical.addContent( online);

    Element primaryURLelem = new Element("primaryURL", defNS);
    online.addContent( primaryURLelem);

    String primaryURL = threddsServerURL +
        "?catalog="+((InvCatalogImpl)ds.getParentCatalog()).getBaseURI().toString() +
        "&dataset="+ds.getID();
    primaryURLelem.addContent( primaryURL);

    Element mediums = new Element("mediums", defNS);
    online.addContent( mediums);
    mediums.addContent( new Element("medium", defNS).addContent("text/html"));

    Element reqs = new Element("requirements", defNS);
    online.addContent( reqs);

    Element req = new Element("requirement", defNS);
    reqs.addContent( req);

    req.addContent( new Element("reqType", defNS).addContent("DLESE:Browser:Netscape"));
    // req.addContent( new Element("reqType", defNS).addContent("DLESE:Browser:Internet Explorer"));

    // educational
    Element educational = new Element("educational", defNS);
    rootElem.addContent( educational);

    Element audiences = new Element("audiences", defNS);
    educational.addContent( audiences);

    Element audience = new Element("audience", defNS);
    audiences.addContent( audience);
    audience.addContent( new Element("gradeRange", defNS).addContent("DLESE:Graduate or professional"));

    Element resourceTypes = new Element("resourceTypes", defNS);
    educational.addContent( resourceTypes);
    String resourceType = "DLESE:Data:In situ dataset"; // default
    if (ds.getDataType() == DataType.GRID)
      resourceType = "DLESE:Data:Modeled dataset"; // take a guess
    else if (ds.getDataType() == DataType.IMAGE)
      resourceType = "DLESE:Data:Remotely sensed dataset";
    resourceTypes.addContent( new Element("resourceType", defNS).addContent(resourceType));

    // rights
    Element rights = new Element("rights", defNS);
    rootElem.addContent( rights);

    rights.addContent( new Element("cost", defNS).addContent("DLESE:No"));
    rights.addContent( new Element("description", defNS).addContent(ds.getDocumentation("rights")));

    /* geospatial
    Element geos = new Element("geospatialCoverages", defNS);
    rootElem.addContent( geos);

    Element geo = new Element("geospatialCoverage", defNS);
    geos.addContent( geo);

    Element body = new Element("body", defNS);
    geo.addContent( body);
    body.addContent( new Element("planet", defNS).addContent("Earth"));

    geo.addContent( new Element("geodeticDatumGlobalOrHorz", defNS).addContent("DLESE:Unknown"));
    geo.addContent( new Element("projection", defNS).addAttribute("type", "DLESE:Unknown"));
    geo.addContent( new Element("coordinateSystem", defNS).addAttribute("type", "DLESE:Unknown")); */
  }

  protected Element writePublisher(ThreddsMetadata.Source p) {
    Element contributor = new Element("contributor", defNS);
    contributor.setAttribute("role", "Publisher");

    Element organization = new Element("organization", defNS);
    contributor.addContent( organization);
    organization.addContent( new Element("instName", defNS).addContent(p.getNameVocab().getText()));

    if ((p.getUrl() != null) && p.getUrl().length() > 0)
      organization.addContent( new Element("instUrl", defNS).addContent(p.getUrl()));

    organization.addContent( new Element("instEmail", defNS).addContent(p.getEmail()));

    return contributor;
  }


  // test
  private static void doOne( InvCatalogFactory fac, String url) {
    System.out.println("***read "+url);
    try {
        InvCatalogImpl cat = (InvCatalogImpl) fac.readXML(url);
        StringBuffer buff = new StringBuffer();
        boolean isValid = cat.check( buff, false);
        System.out.println("catalog <" + cat.getName()+ "> "+ (isValid ? "is" : "is not") + " valid");
        System.out.println(" validation output=\n" + buff);
        // System.out.println(" catalog=\n" + fac.writeXML(cat));

        ADNWriter w = new ADNWriter();
        StringBuffer sbuff = new StringBuffer();
        w.writeDatasetEntries( cat, "C:/data/adn", sbuff);
        System.out.println(" messages=\n"+sbuff);

      } catch (Exception e) {
        e.printStackTrace();
      }

  }

   /** testing */
  public static void main (String[] args) throws Exception {
    InvCatalogFactory catFactory = InvCatalogFactory.getDefaultFactory(true);

    // doOne(catFactory, "file:///C:/dev/thredds/catalog/test/data/TestHarvest.xml");
    doOne(catFactory, "http://motherlode.ucar.edu:8088/thredds/catalog.xml");
    //doOne(catFactory, "http://motherlode.ucar.edu:8088/thredds/configCatGenModels.xml");
  }

}

/* Change History:
   $Log: ADNWriter.java,v $
   Revision 1.3  2004/09/24 03:26:28  caron
   merge nj22

   Revision 1.2  2004/06/09 00:27:26  caron
   version 2.0a release; cleanup javadoc

 */