/*
 *                    BioJava development code
 *
 * This code may be freely distributed and modified under the
 * terms of the GNU Lesser General Public Licence.  This should
 * be distributed with the code.  If you do not have a copy,
 * see:
 *
 *      http://www.gnu.org/copyleft/lesser.html
 *
 * Copyright for this code is held jointly by the individual
 * authors.  These should be listed in @author doc comments.
 *
 * For more information on the BioJava project and its aims,
 * or to join the biojava-l mailing list, visit the home page
 * at:
 *
 *      http://www.biojava.org/
 *
 */
 
package org.biojava.bio.seq;

import java.util.*;
import java.lang.reflect.*;

import org.biojava.utils.*;
import org.biojava.bio.*;
import org.biojava.bio.seq.impl.*;
import org.biojava.bio.seq.projection.*;
import org.biojava.bio.symbol.*;

/**
 * Helper class for projecting Feature objects into an alternative
 * coordinate system.  This class offers a view onto a set of features,
 * projecting them into a different coordinate system, and also changing
 * their <code>parent</code> property.  The destination coordinate system
 * can run in the opposite direction from the source, in which case the
 * <code>strand</code> property of StrandedFeatures is flipped.
 *
 * <p>
 * The projected features returned by this class are small proxy objects.
 * Proxy classes are autogenerated on demand for any sub-interface of
 * <code>Feature</code> by the <code>ProjectionEngine</code> class.
 * </p>
 *
 * @author Thomas Down
 * @author Matthew Pocock
 * @since 1.1
 */

public class ProjectedFeatureHolder extends AbstractFeatureHolder implements FeatureHolder, ProjectionContext {
    private final FeatureHolder wrapped;
    private final FeatureHolder parent;
    private final int translate;
    private final boolean oppositeStrand;
    private ChangeListener underlyingFeaturesChange;
    private Map forwardersByFeature = new HashMap();
    private FeatureHolder topLevelFeatures;

    /**
     * Construct a new FeatureHolder which projects a set of features
     * into a new coordinate system.  If <code>translation</code> is 0
     * and <code>oppositeStrand</code> is <code>false</code>, the features
     * are simply reparented without any transformation.
     *
     * @param fh The set of features to project.
     * @param filter A FeatureFilter to apply to the set of features before projection.
     * @param parent The FeatureHolder which is to act as parent
     *               for the projected features.
     * @param translation The translation to apply to map locations into
     *                    the projected coordinate system.  This is the point
     *                    in the destination coordinate system which is equivalent
     *                    to 0 in the source coordinate system.
     * @param oppositeStrand <code>true</code> if translating into the opposite coordinate system.
     *                       This alters the transformation applied to locations, and also flips
     *                       the <code>strand</code> property of StrandedFeatures.
     * @deprecated Now just wraps up a LazyFilterFeatureHolder
     */

    public ProjectedFeatureHolder(FeatureHolder fh,
				                  FeatureFilter filter,
                                  FeatureHolder parent, 
                                  int translation,
                                  boolean oppositeStrand) 
    {
        this(new LazyFilterFeatureHolder(fh, filter),
             parent,
             translation,
             oppositeStrand);
    }

    /**
     * Construct a new FeatureHolder which projects a set of features
     * into a new coordinate system.  If <code>translation</code> is 0
     * and <code>oppositeStrand</code> is <code>false</code>, the features
     * are simply reparented without any transformation.
     *
     * @param fh The set of features to project.
     * @param parent The FeatureHolder which is to act as parent
     *               for the projected features.
     * @param translation The translation to apply to map locations into
     *                    the projected coordinate system.  This is the point
     *                    in the destination coordinate system which is equivalent
     *                    to 0 in the source coordinate system.
     * @param oppositeStrand <code>true</code> if translating into the opposite coordinate system.
     *                       This alters the transformation applied to locations, and also flips
     *                       the <code>strand</code> property of StrandedFeatures.
     */

    public ProjectedFeatureHolder(FeatureHolder fh,
				                  FeatureHolder parent, 
                                  int translation,
                                  boolean oppositeStrand) 
    {
        this.wrapped = fh;
        this.parent = parent;
        this.translate = translation;
        this.oppositeStrand = oppositeStrand;

        underlyingFeaturesChange = new ChangeListener() {
            public void preChange(ChangeEvent e)
                throws ChangeVetoException 
            {
                if (hasListeners()) {
                    ChangeEvent cev2 = forwardChangeEvent(e);
                    if (cev2 != null) {
                        getChangeSupport(FeatureHolder.FEATURES).firePreChangeEvent(cev2);
                    }
                }
            }

            public void postChange(ChangeEvent e) {
               if (hasListeners()) {
                   ChangeEvent cev2 = forwardChangeEvent(e);
                    if (cev2 != null) {
                        getChangeSupport(FeatureHolder.FEATURES).firePostChangeEvent(cev2);
                    }
               }
            }
        } ;

        getWrapped().addChangeListener(underlyingFeaturesChange);
    }
    
    private FeatureHolder getTopLevelFeatures() {
        if (topLevelFeatures == null) {
            topLevelFeatures = makeProjectionSet(getWrapped());
        }
        return topLevelFeatures;
    }
    
    protected FeatureHolder getWrapped() {
      return wrapped;
    }
    
    //
    // Normal FeatureHolder methods get delegated to our top-level ProjectionSet
    //
    
    public Iterator features() {
        return getTopLevelFeatures().features();
    }
    
    public int countFeatures() {
        return getTopLevelFeatures().countFeatures();
    }
    
    public boolean containsFeature(Feature f) {
        return getTopLevelFeatures().containsFeature(f);
    }
    
    public FeatureHolder filter(FeatureFilter ff) {
        return getTopLevelFeatures().filter(ff);
    }
    
    public FeatureHolder filter(FeatureFilter ff, boolean recurse) {
        return getTopLevelFeatures().filter(ff, recurse);
    }
    
    public Feature createFeature(Feature.Template templ) 
	        throws ChangeVetoException, BioException
    {
        throw new ChangeVetoException("Can't create features in this projection");
    }

    public void removeFeature(Feature f) 
        throws ChangeVetoException
	{
        throw new ChangeVetoException("Can't create features in this projection");
	}
    
    public FeatureFilter getSchema() {
        return getTopLevelFeatures().getSchema();
    }
    
    //
    // Dumb set of features to which we delegate everything except the
    // ChangeEvent stuff.
    //
    
    private class ProjectionSet implements FeatureHolder {
        private final FeatureHolder baseSet;
        
        ProjectionSet(FeatureHolder baseSet) {
            this.baseSet = baseSet;
        }
        
        public int countFeatures() {
            return baseSet.countFeatures();
        }

        public Iterator features() {
            final Iterator wrappedIterator = baseSet.features();
            return new Iterator() {
                public boolean hasNext() {
                    return wrappedIterator.hasNext();
                }
            
                public Object next() {
                    return projectFeature((Feature) wrappedIterator.next());
                }
            
                public void remove() {
                    throw new UnsupportedOperationException();
                }
            } ;
        }
    
        public boolean containsFeature(Feature f) {
            for (Iterator fi = features(); fi.hasNext(); ) {
                if (f.equals(fi.next())) {
                    return true;
                }
            }
            return false;
        }

        public FeatureHolder filter(FeatureFilter ff) {
            return filter(ff, true); // bit of a hack for now.
        }
    
        public FeatureHolder filter(FeatureFilter ff, boolean recurse) {
            ff = untransformFilter(ff);
            FeatureHolder toProject = baseSet.filter(ff, recurse);
            return makeProjectionSet(toProject);
        }
    
        public Feature createFeature(Feature.Template templ) 
	        throws ChangeVetoException
        {
            throw new ChangeVetoException("Can't create features in this projection");
        }

        public void removeFeature(Feature f) 
                throws ChangeVetoException
	    {
	        throw new ChangeVetoException("Can't create features in this projection");
	    }
        
        public FeatureFilter getSchema() {
            return transformFilter(baseSet.getSchema());
        }
        
        public void addChangeListener(ChangeListener cl) {}
        public void removeChangeListener(ChangeListener cl) {}
        public void addChangeListener(ChangeListener cl, ChangeType ct) {}
        public void removeChangeListener(ChangeListener cl, ChangeType ct) {}
        public boolean isUnchanging(ChangeType ct) { return true; }
    }
    
    /**
     * Called to transform a FeatureFilter applying to our projections into a corresponding
     * filter on the parent FeatureHolder.
     *
     * @since 1.3
     */
    
    protected FeatureFilter untransformFilter(FeatureFilter ff) {
        return FilterUtils.transformFilter(
            ff,
            new FilterUtils.FilterTransformer() {
                public FeatureFilter transform(FeatureFilter ff) {
                    if (ff instanceof FeatureFilter.OverlapsLocation) {
                        return new FeatureFilter.OverlapsLocation(untransformLocation(((FeatureFilter.OverlapsLocation) ff).getLocation()));
                    } else if (ff instanceof FeatureFilter.ContainedByLocation) {
                        return new FeatureFilter.ContainedByLocation(untransformLocation(((FeatureFilter.ContainedByLocation) ff).getLocation()));
                    } else if (ff instanceof FeatureFilter.StrandFilter) {
                        return new FeatureFilter.StrandFilter(transformStrand(((FeatureFilter.StrandFilter) ff).getStrand()));
                    } else {
                        return ff;
                    }
                }   
            }
        ) ;
    }
    
    /**
     * Called to transform a FeatureFilter applying to our parent FeatureHolder into the
     * coordinate system of our parent.
     *
     * @since 1.3
     */
    
    protected FeatureFilter transformFilter(FeatureFilter ff) {
        return FilterUtils.transformFilter(
            ff,
            new FilterUtils.FilterTransformer() {
                public FeatureFilter transform(FeatureFilter ff) {
                    if (ff instanceof FeatureFilter.OverlapsLocation) {
                        return new FeatureFilter.OverlapsLocation(transformLocation(((FeatureFilter.OverlapsLocation) ff).getLocation()));
                    } else if (ff instanceof FeatureFilter.ContainedByLocation) {
                        return new FeatureFilter.ContainedByLocation(transformLocation(((FeatureFilter.ContainedByLocation) ff).getLocation()));
                    } else if (ff instanceof FeatureFilter.StrandFilter) {
                        return new FeatureFilter.StrandFilter(transformStrand(((FeatureFilter.StrandFilter) ff).getStrand()));
                    } else {
                        return ff;
                    }
                }   
            }
        ) ;
    }
    
    /**
     * Called to transform a strand property between projection and underlying coordinates.
     */
    
    protected StrandedFeature.Strand transformStrand(StrandedFeature.Strand strand) {
        if (oppositeStrand) {
            return strand.flip();
        } else {
            return strand;
        }
    }
    
    /**
     * Called to transform a location from underlying to projection coordinates.
     */
    
    protected Location transformLocation(Location oldLoc) {
        return ProjectionUtils.transformLocation(oldLoc, translate, oppositeStrand);
    }

    /**
     * Called to transform a location from projection to underlying coordinates.
     */
    
    protected Location untransformLocation(Location oldLoc) {
        if (oppositeStrand) {
            if (oldLoc.isContiguous()) {
                if (oldLoc instanceof PointLocation){
                    return new PointLocation(translate - oldLoc.getMin());
                } else {
                    return new RangeLocation(translate - oldLoc.getMax(),
    	                                     translate - oldLoc.getMin());
                }
            } else {
                Location compound = Location.empty;
                List locList = new ArrayList();
                for (Iterator i = oldLoc.blockIterator(); i.hasNext(); ) {
                    Location oldBlock = (Location) i.next();
                    locList.add(new RangeLocation(translate - oldBlock.getMax(),
                    		      			translate - oldBlock.getMin()));
                }
                compound = LocationTools.union(locList);
                return compound;
            }
        } else {
            return oldLoc.translate(-translate);
        }
    }
    
    /**
     * Create a single projected feature using the rules of this <code>ProjectedFeatureHolder</code>.
     */
    
    public Feature projectFeature(Feature f) {
        return ProjectionEngine.DEFAULT.projectFeature(f, this);
    }

    /**
     * Return the translation component of the transformation applied by this FeatureHolder
     */
    
    public int getTranslation() {
        return translate;
    }

    /**
     * Return true if projected features should be flipped to the opposite strand
     */
    
    public boolean isOppositeStrand() {
        return oppositeStrand;
    }

    /**
     * Return the parent of all top-level features in this FeatureHolder.
     */
    
    public FeatureHolder getParent() {
        return parent;
    }
    
    /**
     * Called internally to construct a lightweight projected view of a set of features
     */
    
    protected FeatureHolder makeProjectionSet(FeatureHolder fh) {
        return new ProjectionSet(fh);
    }
    
    //
    // The following methods are our implementation of ProjectionContext
    //
    
        public FeatureHolder getParent(Feature f) {
            FeatureHolder oldP = f.getParent();
            if (oldP instanceof Feature) {
                if (wrapped.containsFeature(f)) {
                    return parent;
                } else {
                    return projectFeature((Feature) oldP);
                }
            } else {
                return parent;
            }
        }	    

        public Sequence getSequence(Feature f) {
            FeatureHolder fh = parent;
            while (fh instanceof Feature) {
                fh = ((Feature) fh).getParent();
            }
            return (Sequence) fh;
        }

        public Location getLocation(Feature f) {
            Location oldLoc = f.getLocation();
            return transformLocation(oldLoc);
        }

        public StrandedFeature.Strand getStrand(StrandedFeature sf) {
            StrandedFeature.Strand s = sf.getStrand();
            return transformStrand(s);
        }

        public Annotation getAnnotation(Feature f) {
            return f.getAnnotation();
        }

        public FeatureHolder projectChildFeatures(Feature f, FeatureHolder parent) {
            return makeProjectionSet(f);
        }

        public Feature createFeature(Feature f, Feature.Template templ) 
	        throws BioException, ChangeVetoException
        {
            throw new ChangeVetoException("Can't create features in this projection");
        }

        public void removeFeature(Feature f, Feature f2) 
            throws ChangeVetoException
	    {
	        throw new ChangeVetoException("Can't create features in this projection");
	    }
        
        public FeatureFilter getSchema(Feature f) {
            return transformFilter(f.getSchema());
        }
        
    //
    // Event wiring stuff
    //    
        
    public void addChangeListener(Feature f, ChangeListener cl, ChangeType ct) {
        if (!f.isUnchanging(ct)) {
            PFChangeForwarder forwarder = (PFChangeForwarder) forwardersByFeature.get(f);
            if (forwarder == null) {
                forwarder = new PFChangeForwarder(f);
                forwardersByFeature.put(f, forwarder);
                f.addChangeListener(forwarder, ChangeType.UNKNOWN);
            }
            forwarder.addChangeListener(cl, ct);
        }
    }
        
    public void removeChangeListener(Feature f, ChangeListener cl, ChangeType ct) {
        PFChangeForwarder forwarder = (PFChangeForwarder) forwardersByFeature.get(f);
        if (forwarder != null) {
            forwarder.removeChangeListener(cl, ct);
            if (!forwarder.hasListeners()) {
                forwardersByFeature.remove(f);
                f.removeChangeListener(forwarder, ChangeType.UNKNOWN);
            }
        }
    }
    
    private class PFChangeForwarder extends ChangeSupport implements ChangeListener {
        private Feature master;
        
        public PFChangeForwarder(Feature master) {
            super(1);
            this.master = master;
        }
        
        public void preChange(ChangeEvent cev)
            throws ChangeVetoException
        {
            ChangeEvent cev2 = forwardFeatureChangeEvent(master, cev);
            if (cev2 != null) {
                firePreChangeEvent(cev2);
            }
        }
        
        public void postChange(ChangeEvent cev) {
            ChangeEvent cev2 = forwardFeatureChangeEvent(master, cev);
            if (cev2 != null) {
                firePostChangeEvent(cev2);
            }
        }
    }
        
    /**
     * Called internally to generate a forwarded version of a ChangeEvent from a ProjectedFeature
     *
     * @param f the feature who's projection is due to receive an event.
     * @return a tranformed event, or <code>null</code> to cancel the event.
     */
        
    protected ChangeEvent forwardFeatureChangeEvent(Feature f, ChangeEvent cev) {
        return new ChangeEvent(projectFeature(f),
                               cev.getType(),
                               cev.getChange(),
                               cev.getPrevious(),
                               cev);
    }
    
    /**
     * Called internally to generate a forwarded version of a ChangeEvent from our
     * underlying FeatureHolder
     */
        
    protected ChangeEvent forwardChangeEvent(ChangeEvent cev) {
        return new ChangeEvent(this,
                               cev.getType(),
                               cev.getChange(),
                               cev.getPrevious(),
                               cev);
    }
}
