/*
 *                    BioJava development code
 *
 * This code may be freely distributed and modified under the
 * terms of the GNU Lesser General Public Licence.  This should
 * be distributed with the code.  If you do not have a copy,
 * see:
 *
 *      http://www.gnu.org/copyleft/lesser.html
 *
 * Copyright for this code is held jointly by the individual
 * authors.  These should be listed in @author doc comments.
 *
 * For more information on the BioJava project and its aims,
 * or to join the biojava-l mailing list, visit the home page
 * at:
 *
 *      http://www.biojava.org/
 *
 */

package org.biojava.bio.program.ssbind;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

import org.biojava.bio.search.SearchContentHandler;
import org.biojava.utils.stax.SAX2StAXAdaptor;

/**
 * A <code>SeqSimilarityAdapter</code> converts SAX events into method
 * calls on a <code>SearchContentHandler</code> implementation. The
 * SAX events should describe elements conforming to the BioJava
 * BlastLikeDataSetCollection DTD. A
 * <code>BlastLikeSearchBuilder</code> is supplied, implementing the
 * <code>SearchContentHandler</code> interface, which will create
 * <code>SeqSimilaritySearchResult</code>s from the stream.
 *
 * @author Keith James
 * @since 1.2
 */
public class SeqSimilarityAdapter implements ContentHandler
{
    private SeqSimilarityStAXAdapter staxHandler;
    private ContentHandler contentHandler;
    
    public SeqSimilarityAdapter()
    {
        staxHandler = new SeqSimilarityStAXAdapter();
        contentHandler = new SAX2StAXAdaptor(staxHandler);
    }

    /**
     * <code>getSearchContentHandler</code> gets the handler which
     * will recieve the method calls generated by the adapter.
     *
     * @return a <code>SearchContentHandler</code>.
     */
    public SearchContentHandler getSearchContentHandler()
    {
        return staxHandler.getSearchContentHandler();
    }

    /**
     * <code>setSearchContentHandler</code> sets the handler which
     * will recieve the method calls generated by the adapter.
     *
     * @param scHandler a <code>SearchContentHandler</code>.
     */
    public void setSearchContentHandler(SearchContentHandler scHandler)
    {
        staxHandler.setSearchContentHandler(scHandler);
    }

    public void startDocument() throws SAXException
    {
        contentHandler.startDocument();
    }

    public void endDocument() throws SAXException
    {
        contentHandler.endDocument();
    }

    public void characters(char[] ch, int start, int end) throws SAXException
    {
	contentHandler.characters(ch, start, end);
    }

    public void ignorableWhitespace(char[] ch, int start, int end)
	throws SAXException
    {
	contentHandler.ignorableWhitespace(ch, start, end);
    }

    public void startPrefixMapping(String prefix, String uri)
        throws SAXException
    {
	contentHandler.startPrefixMapping(prefix, uri);
    }

    public void endPrefixMapping(String prefix) throws SAXException
    {
	contentHandler.endPrefixMapping(prefix);
    }

    public void processingInstruction(String target, String data)
        throws SAXException
    {
	contentHandler.processingInstruction(target, data);
    }

    public void setDocumentLocator(Locator locator)
    {
	contentHandler.setDocumentLocator(locator);
    }

    public void skippedEntity(String name) throws SAXException
    {
	contentHandler.skippedEntity(name);
    }

    public void startElement(String     nsURI,
			     String     localName,
			     String     qName,
			     Attributes attrs)
	throws SAXException
    {
        contentHandler.startElement(nsURI, localName, qName, attrs);
    }

    public void endElement(String nsURI, String localName, String qName)
        throws SAXException
    {
        contentHandler.endElement(nsURI, localName, qName);
    }

    /**
     * <code>getProgram</code> returns the program type which
     * generated the results.
     *
     * @return a <code>String</code> indicating the progam
     * name.
     */
    String getProgram()
    {
        return staxHandler.getProgram();
    }

    /**
     * <code>setProgram</code> informs the adapter which program type
     * it is working on.
     *
     * @param program a <code>String</code> indicating the progam
     * name.
     */
    void setProgram(String program)
    {
        staxHandler.setProgram(program);
    }
}
